<?php

/**
 * @param Database : For setting up and maintaining Database Connection
 * @param DatabaseResult : Is a return type for Database methods : not used directly
 * @param HostInfo : Holds information about the current client
 * @param LocationPoint : This packages stores coordinates point
 *
 *
 * @author : Kadian Salmon
 * Purpose : This contains classes for php backend manipulation
 *
 */

/**
 * @param DB_USERNAME : This is the USERNAME for the database
 * @param DB_PASSWORD : This is the PASSWORD for the database
 * @param DB_NAME : This is the NAME for the active database
 * @param DB_HOST : This is the HOST_NAME for the DB server. If None = "localhost"
 *
 * @method0 connect_error : returns any errors encountered during the connection of the database
 * @method0 query($) : processes the passed string sql query and returns a : DatabaseResult Object
 * @method0 getUsername : returns database USERNAME
 * @method0 getPassword : returns database PASSWORD
 * @method0 getDatabase : returns database NAME
 * @method0 getHost : returns database HOST_NAME
 * @method0 getConnector : returns the LINK to the database
 * @method0 transaction : begin a transaction - for queries that are executed as a single transaction
 * @method0 commit : commit a transaction - for queries that are executed as a single transaction
 * @method0 rollback : undo a transaction - in case either of a transaction query fails - for queries that are executed as a single transaction
 * @method0 @setters for above
 *
 * @example-new_instance_1 : $mydb = new Database("minuteman","mintemanid24","secret_info","29.panel.db.com");
 * @example-new_instance_2 : $mydb = new Database("minuteman","mintemanid24","secret_info");
 *
 * @example-select_query_db : $result_s = $mydb->query("SELECT * FROM my_table WHERE u_id = 'jhfdghvdsgdsv' "); : DatabaseResult
 * @example-check_result : $result_s->is_row_found() : Boolean
 * @example-view_result_1 : $result_s->fetch_array() : Associated array
 * @example-view_result_2 : while($row = $result_s->fetch_array()) : Associated array
 *
 * @example-update_query_db : $result_u = $mydb->query("UPDATE my_table SET age = 22 WHERE u_id = 'jhfdghvdsgdsv' "); : DatabaseResult
 * @example-check_result : $result_u->is_row_affected() : Boolean
 * @subpackage : Database
 *   Date : Sep 18, 2019
 *   Date Updated : Sep 18, 2019
 * @uses : Maintains communication to a defined mysql database
 */
class Database
{

    private $username;
    private $password;
    private $database;
    private $host;
    private $connector;

    public function __construct($username = "", $password = "", $database = "", $host = "")
    {
        $this->username = $username;
        $this->password = $password;
        $this->database = $database;
        $this->host = $host;

        if ($this->host == "") {
            $this->host = "localhost";
        }

        $this->connector = mysqli_connect($this->host, $this->username, $this->password, $this->database);
        $this->connector->set_charset("utf8mb4"); //To support Emoji / Special Characters / Asian Characters

    }

    public function connect_error()
    {

        return mysqli_connect_errno();
    }


    public function query($sql)
    {

        /*
          mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);//must be called in order to enforce try error block
            try {
            $query_result = mysqli_query($this->connector, $sql);

        }
        catch (Exception $exc) {
            echo 'dsjbhubwsjb';
            return null;
        }*/
        $query_result = mysqli_query($this->connector, $sql);

        if ($query_result != false) {
            return new DatabaseResult($query_result, $this->connector);
        }

        return null;//false is returned on error not empty result

    }

    /**
     * @param $sql
     * @return bool
     */
    public function update($sql): bool
    {
        $result = $this->query($sql);
        return (!is_null($result) && $result->is_row_affected());
    }

    /**
     * @param $sql
     * @return bool
     */
    public function insert($sql): bool
    {
        $result = $this->query($sql);
        return (!is_null($result) && $result->is_row_affected());
    }

    public function fetch_assoc($sql)
    {
        $result = $this->query($sql);
        if (!is_null($result)) return $result->fetch_assoc();
        return null;
    }

    public function fetch_object($sql)
    {
        $result = $this->query($sql);
        if (!is_null($result)) return $result->fetch_object();
        return null;
    }

    /**
     * @param $sql
     * @return false|int|string|null
     */
    public function get_insert_id($sql)
    {
        $result = $this->query($sql);

        if (!is_null($result) && $result->is_row_affected()) {
            return $result->insert_id();
        }
        return null;
    }

    public function getUsername()
    {
        return $this->username;
    }

    public function getPassword()
    {
        return $this->password;
    }

    public function getDatabase()
    {
        return $this->database;
    }

    public function getHost()
    {
        return $this->host;
    }

    public function getConnector()
    {
        return $this->connector;
    }

    public function setUsername($username)
    {
        $this->username = $username;
    }

    public function setPassword($password)
    {
        $this->password = $password;
    }

    public function setDatabase($database)
    {
        $this->database = $database;
    }

    public function setHost($host)
    {
        $this->host = $host;
    }

    public function setConnector($connector)
    {
        $this->connector = $connector;
    }

    /**
     * @return bool
     */
    public function transaction(): bool
    {
        return $this->connector->begin_transaction();
    }

    /**
     * @return bool
     */
    public function commit(): bool
    {
        return $this->connector->commit();
    }


    /**
     * @return bool
     */
    public function rollback(): bool
    {
        return $this->connector->rollback();
    }

    /**
     * @return string
     */
    function get_error(): string
    {
//        mysqli_error();
        return mysqli_error($this->connector);
    }
}

/**
 * @param DB_PASSWORD : This is the PASSWORD for the database
 * @param DB_NAME : This is the NAME for the active database
 * @param DB_HOST : This is the HOST_NAME for the DB server. If None = "localhost"
 *
 * @method0 fetch_array : return results in the form of an associated array
 * @method0 rows_count : returns a count after a select query
 * @method0 rows_affected : returns a count after an update query
 * @method0 is_row_affected : returns true if a row was affected after an update query
 * @method0 is_row_found : returns true if 1 or more records were found after a select query
 * @method0 insert_id : returns the last inserted id if success or false if failed
 * @method0 get_result : returns the raw db stream to be used in a mysql func => mysqli_affected_rows(result_here)
 * @method0 set_result($) : sets the raw db stream
 * @method0 fetch_assoc : return results in the form of an associated array
 * @method0 fetch_object : return results of a single row as an object
 *
 * @subpackage @internal : This is the USERNAME for the database
 *   Date : Sep 18, 2019
 */
class DatabaseResult
{

    private $query_result;
    private $connector;

    public function __construct($query_result, $conn)
    {
        $this->query_result = $query_result;

        $this->connector = $conn;
    }

    public function fetch_array()
    {
        return mysqli_fetch_array($this->query_result);
    }

    public function fetch_field()
    {
        return mysqli_fetch_fields($this->query_result);
    }

    public function rows_count()
    {
        //this can be use after a select query
        return mysqli_num_rows($this->query_result);//number of rows returned
    }

    public function rows_affected()
    {
        //this can be used after an update query
        return mysqli_affected_rows($this->connector);//number of rows changed
    }

    public function fetch_object()
    {
        return mysqli_fetch_object($this->query_result);
    }

    public function fetch_assoc()
    {
        return mysqli_fetch_assoc($this->query_result);
    }

    public function is_row_affected()
    {
        //this can be used after an update query
        if (mysqli_affected_rows($this->connector) > 0) {
            //yes indeed ; rows were affected

            return true;
        }

        return false;
    }

    public function is_row_found()
    {
        //this can be use after a select query
        if (mysqli_num_rows($this->query_result) > 0) {
            //yes indeed ; rows were affected

            return true;
        }

        return false;
    }

    public function insert_id()
    {
        //get the last inserted id of the table has auto increment

        $last_id = mysqli_insert_id($this->connector);
        //this can be use after a insert query
        if ($last_id != false) {
            //yes indeed ; rows were affected

            return $last_id;
        }

        return false;
    }

    function get_result()
    {
        return $this->query_result;
    }

    function set_result($query_result)
    {
        $this->query_result = $query_result;
    }
}


/**
 * @param DB_HANDLE : This is a handle to a Database instance
 * @return ip_address|country_name|country_code|area_code|currency
 *
 *
 * @method0 getIp_address : returns the ip address of this client => x.x.x.x
 * @method0 getCountry_name : returns the country name for this client => jamaica
 * @method0 getCountry_code : returns the 2 character country code for client => JM
 * @method0 getArea_code : returns the area code => 1876
 * @method0 getCurrency : returns the currency for that country => JMD
 * @method0 getDb : returns the handle for the database that contains the IP-RANGE => COUNTRY map
 * @method @internal set_host_data() : set host properties based on host data
 * @method @internal get_ip() : returns the client ip to be used by internal methods
 * @method @internal get_data_from_ip($) : this logic pulls db data based on the client's ip address
 * @method @internal @return get_data_from_ip($) : {
 * country_name|country_code|area_code|currency
 * }
 *
 * @NOTE get_data_from_ip - [its currently rigged]
 *
 * @how_to_use
 * $client = new HostInfo($db);
 * $ip = $_REQUEST["ip"];
 * if($ip != "my_ip"){
 * $client->setIp_address($ip);
 * }
 * @subpackage : HostInfo
 *   Date : Oct 25, 2019
 * @uses : This packages information related to the connected Host
 *
 */
class HostInfo
{
    private $API_IP_QUERY = "https://api.queritel.com/api/localization/query/ip_query.php";

    private $ip_address = null;//x.x.x.x
    private $country_name = null;//jamaica
    private $country_code = null;//JM
    private $country_flag = null;//jm.svg
    private $area_code = null;//1876
    private $currency = null;//JMD
    private $db = null;//db connector

    public function __construct(Database $db = null)
    {
        $this->db = $db;

        $this->set_host_data();//assign data to variables
    }


    public function getIp_address()
    {
        return $this->ip_address;
    }

    public function getCountry_name()
    {
        return $this->country_name;
    }

    public function getCountry_code()
    {
        return $this->country_code;
    }

    public function getCountry_flag()
    {
        return $this->country_flag;
    }

    public function getArea_code()
    {
        return $this->area_code;
    }

    public function getCurrency()
    {
        return $this->currency;
    }

    public function getDb()
    {
        return $this->db;
    }

    public function setIp_address($ip_address)
    {
        $this->ip_address = $ip_address;
        $this->set_host_data();
    }

    public function default_ip()
    {
        $this->ip_address = $this->get_ip();
        $this->set_host_data();
    }

    public function setCountry_name($country_name)
    {
        $this->country_name = $country_name;
    }

    public function setCountry_code($country_code)
    {
        $this->country_code = $country_code;
    }

    public function setCountry_flag($country_flag)
    {
        $this->country_flag = $country_flag;
    }

    public function setArea_code($area_code)
    {
        $this->area_code = $area_code;
    }

    public function setCurrency($currency)
    {
        $this->currency = $currency;
    }

    public function setDb($db)
    {
        $this->db = $db;
    }

    //todo : fix currency support
    private function set_host_data()
    {
        if ($this->ip_address == null) {
            $this->ip_address = $this->get_ip();
        }

        $host_detail = $this->get_data_from_ip($this->ip_address);

        $this->country_name = $host_detail["country_name"];
        $this->country_code = $host_detail["country_code"];
        $this->country_code = $host_detail["country_code"];
        $this->country_flag = $host_detail["country_flag"];

        switch ($this->country_code) {
            case "CL" :
                $this->currency = "CLP";
                break;

            case "JM" :
                $this->currency = "JMD";
                break;

            default :
                $this->currency = "USD";
                break;
        }


    }

    private function get_ip()
    {
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {//check ip from share internet
            $ip = $_SERVER['HTTP_CLIENT_IP'];
        } else {
            $ip = $_SERVER['REMOTE_ADDR'];
        }
        return $ip;
    }

    private function get_data_from_ip($ip)
    {
        //use default
        //$data["country_name"] = "Jamaica";
        //$data["country_code"] = "JM";

        $url = $this->API_IP_QUERY . "?ip=" . $ip;

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);

        curl_setopt($ch, CURLOPT_HTTPHEADER, array('Accept: application/json'));

        $result = curl_exec($ch);
        curl_close($ch);

        $out = array();
        $data = json_decode($result, true);


        $out["country_name"] = $data["country_name"];
        $out["country_code"] = $data["country_code"];
        $out["country_flag"] = $data["country_flag"];


        return $out;

    }

}

/**
 * @param $latitude : This is the latitude
 * @param $longitude : This is the longitude
 * @return $latitude | $longitude
 *
 *
 * @method0 setLatitude : set latitude for point
 * @method0 setLongitude : set longitude for point
 * @method0 getLatitude : get latitude for point
 * @method0 getLongitude : get longitude for point
 * @method0 isEmpty : @return true if the object is empty
 *
 * @uses : This packages stores coordinate point
 *
 * @subpackage : LocationPoint
 *   Date : Oct 25, 2019
 */
class LocationPoint
{

    private $latitude = null;
    private $longitude = null;

    public function __construct($latitude = null, $longitude = null)
    {
        $this->latitude = $latitude;
        $this->longitude = $longitude;
    }

    public function setLatitude($latitude)
    {
        $this->latitude = $latitude;
    }

    public function setLongitude($longitude)
    {
        $this->longitude = $longitude;
    }

    public function getLatitude()
    {
        return $this->latitude;
    }

    public function getLongitude()
    {
        return $this->longitude;
    }

    public function isEmpty()
    {
        if ($this->latitude == null || $this->longitude == null) {
            return true;
        }

        return false;
    }
}
